import json
import os
import genanki
import re
from html import unescape

# === PARAMÈTRES ===
DOSSIER_BASE = os.path.dirname(os.path.abspath(__file__))
JSON_FILE = os.path.join(DOSSIER_BASE, "donnees.json")
DOSSIER_IMAGES = os.path.join(DOSSIER_BASE, "fichiers")
OUTPUT_APKG = os.path.join(DOSSIER_BASE, "Sciences_physiques_au_college.apkg")

# === CONFIGURATION DU DECK ===
DECK_ID = 20251026
MODEL_ID = 20251027
deck_name = "Sciences_physiques_au_college"

# === FONCTION DE NETTOYAGE ===
def clean_text(t):
    """Nettoie le texte, conserve HTML utile et convertit en LaTeX Anki."""
    if not t:
        return ""
    t = unescape(t)
    t = t.replace("ρ", "\\( \\rho \\)")
    t = t.replace("→", "\\( \\rightarrow \\)")
    t = re.sub(r"([A-Z][a-z]?)(\d+)", r"\1_{\2}", t)  # indices chimiques
    return t.strip()

# === CHARGEMENT DES DONNÉES ===
with open(JSON_FILE, "r", encoding="utf-8") as f:
    data = json.load(f)
cartes = data.get("cartes", [])

# === MODÈLE ANKI ===
model = genanki.Model(
    MODEL_ID,
    "Modèle Sciences Physiques",
    fields=[
        {"name": "Question"},
        {"name": "Réponse"},
    ],
    templates=[
        {
            "name": "Carte 1",
            "qfmt": "{{Question}}",
            "afmt": "{{FrontSide}}<hr id='answer'>{{Réponse}}",
        },
    ],
    css="""
    .card {
        font-family: Arial;
        font-size: 18px;
        text-align: left;
        color: black;
        background-color: white;
    }
    img { max-height: 300px; }
    """
)

# === CRÉATION DU DECK ===
deck = genanki.Deck(DECK_ID, deck_name)
media_files = []

# === GÉNÉRATION DES CARTES ===
for c in cartes:
    recto = clean_text(c["recto"].get("texte", ""))
    verso = clean_text(c["verso"].get("texte", ""))

    img_recto = c["recto"].get("image", "")
    img_verso = c["verso"].get("image", "")

    if img_recto:
        recto += f'<br><img src="{img_recto}">'
        media_files.append(os.path.join(DOSSIER_IMAGES, img_recto))
    if img_verso:
        verso += f'<br><img src="{img_verso}">'
        media_files.append(os.path.join(DOSSIER_IMAGES, img_verso))

    note = genanki.Note(model=model, fields=[recto, verso])
    deck.add_note(note)

# === PAQUET FINAL ===
package = genanki.Package(deck)
package.media_files = media_files
package.write_to_file(OUTPUT_APKG)

print(f"✅ Fichier Anki généré : {OUTPUT_APKG}")
print(f"📸 Nombre d’images intégrées : {len(media_files)}")
print(f"🃏 Nombre total de cartes : {len(cartes)}")
